/*----------------------------------------------------------------------------
 * Name:    bc7701.c
 * Purpose:
 * Note(s):
 *----------------------------------------------------------------------------
 * This file is part of the uVision/ARM development tools.
 * This software may only be used under the terms of a valid, current,
 * end user licence from KEIL for a compatible version of KEIL software
 * development tools. Nothing else gives you the right to use this software.
 *
 * This software is supplied "AS IS" without warranties of any kind.
 *
 * Copyright (c) 2012 Keil - An ARM Company. All rights reserved.
 *----------------------------------------------------------------------------*/

#include    <stdlib.h>
#include		<string.h>
#include		"timer.h"
#include		"bc7701.h"
#include		"hci.h"

uc32	BaudRateTable[]={	9600,14400,19200,38400,57600,115200,125000 };
uc8	DummyWakeUpWidth[] = { 1, 1, 2, 2, 3, 4, 6 }; 
uc8	DummyWakeupPackage[] = { 0xFF,4,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };

static u8	UART_BaudRate;
vu8	uRxSerialBuffer[UART_BUFFER_SIZE];
static u8	uRxSBWriteIdx,uRxSBReadIdx;

#define	TXSERIAL_QUEUE_MAX		16
void *TxSerialQueue[TXSERIAL_QUEUE_MAX];
vu8	uTxQWriteIdx,uTxQReadIdx;

static void	*uTxPackageIdx;
static vu8	uTxPackageLength,uTxPackagePos;

#define	RXSERIAL_QUEUE_MAX		16
void *RxSerialQueue[RXSERIAL_QUEUE_MAX];
vu8	uRxQWriteIdx,uRxQReadIdx;

static u8	*uRxPkgIndex;
static vu8	uRxPkgStep;
static vu8	uRxPkgType,uRxPkgEvent,uRxPkgLength,uRxPkgPosint;

//#define IS_TXD_BUF_FULL(LEN)		(((uTxWriteIdx + LEN) >= UART_BUFFER_SIZE) ? ((uTxWriteIdx + LEN - UART_BUFFER_SIZE) == uTxReadIdx) : ((uTxWriteIdx + LEN) == uTxReadIdx))
//#define IS_TXD_BUF_EMPTY()      	(uTxReadIdx == uTxWriteIdx)
#define IS_TXD_QUE_FULL(LEN)		(((uTxQWriteIdx + LEN) >= TXSERIAL_QUEUE_MAX) ? ((uTxQWriteIdx + LEN - TXSERIAL_QUEUE_MAX) == uTxQReadIdx) : ((uTxQWriteIdx + LEN) == uTxQReadIdx))
#define IS_TXD_QUE_EMPTY()      	(uTxQReadIdx == uTxQWriteIdx)
#define IS_RXD_BUF_FULL(LEN)		(((uRxSBWriteIdx + LEN) >= UART_BUFFER_SIZE) ? ((uRxSBWriteIdx + LEN - UART_BUFFER_SIZE) == uRxSBReadIdx) : ((uRxSBWriteIdx + LEN) == uRxSBReadIdx))
#define IS_RXD_BUF_EMPTY()      	(uRxSBReadIdx == uRxSBWriteIdx)
#define IS_RXD_QUE_FULL(LEN)		(((uRxQWriteIdx + LEN) >= RXSERIAL_QUEUE_MAX) ? ((uRxQWriteIdx + LEN - RXSERIAL_QUEUE_MAX) == uRxQReadIdx) : ((uRxQWriteIdx + LEN) == uRxQReadIdx))
#define IS_RXD_QUE_EMPTY()      	(uRxQReadIdx == uRxQWriteIdx)

/*------------------------------------------------------------------------------
* Function Name : BC7701_InterfaceConfigure
* Description   : setup UART port & RESET pin for IO port
* Input         :
*				- br : UART Baud rate
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_InterfaceConfigure(u8 br)
{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
	/* Enable peripheral clock of BLE UART */
	UART_CLK(CKCUClock) = 1;
	CKCU_PeripClockConfig(CKCUClock, ENABLE);

	/* Config AFIO mode as UART_RXD and UART_TXD function. */
	AFIO_GPxConfig(UART_TXD_AFIO_PORT, UART_TXD, UART_TXD_AFIO_FUN);
	AFIO_GPxConfig(UART_RXD_AFIO_PORT, UART_RXD, UART_RXD_AFIO_FUN);

	BC7701_UARTConfigure(br);

	/* Config AFIO mode as RESET_PIN function. */
	AFIO_GPxConfig(RESET_AFIO_PORT, RESET_PIN, RESET_AFIO_FUN);
	GPIO_DirectionConfig(RESET_PIN_PORT,RESET_PIN,GPIO_DIR_OUT);
	BC7701_RESET_CLR();
	uRxSBWriteIdx=uRxSBReadIdx=0;	
	uTxQWriteIdx=uTxQReadIdx=0;
	uRxQWriteIdx=uRxQReadIdx=0;
	uRxPkgStep = PARS_PKG_TYPE;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_UARTConfigure
* Description   : UART configure
*				- BaudRate = br
*				- Word Length = 8 Bits
*				- One Stop Bit
*				- None parity bit
* Input         :
*				- br : UART Baud rate
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_UARTConfigure(u8 br)
{
	USART_InitTypeDef USART_InitStructure;

	UART_BaudRate = br;
	/* UART configuration -----------------------------------*/
	USART_InitStructure.USART_BaudRate = BaudRateTable[UART_BaudRate];
	USART_InitStructure.USART_WordLength = UART_WORD_LENGTH;
	USART_InitStructure.USART_StopBits = UART_STOP_BIT;
	USART_InitStructure.USART_Parity = UART_PARITY;
	USART_InitStructure.USART_Mode = UART_MODE;
	USART_Init(UART_PORT,&USART_InitStructure);

	/* Seting UART_PORT interrupt-flag */
	USART_IntConfig(UART_PORT, USART_INT_RXDR, ENABLE);
	/* Enable UART_PORT */
	USART_TxCmd(UART_PORT, ENABLE);
	USART_RxCmd(UART_PORT, ENABLE);
	/* Enable UART_PORT interrupt */
	NVIC_EnableIRQ(UART_IRQn);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_RESET_SET
* Description   : BC7701 reset pin to high
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_RESET_SET(void)
{
	RESET_PIN_PORT->SRR = RESET_PIN;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_RESET_CLR
* Description   : BC7701 reset pin to low
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_RESET_CLR(void)
{
	RESET_PIN_PORT->RR = RESET_PIN;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_HardwareBuadRateDefault
* Description   : Hardware definition BC 7701 baud rate
* Input         :
* 			- br : buard rate
* 				- 9600 : TXD=1,RXD=0
* 				- Other : TXD=1,RXD=1 to define 115200
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_HardwareBuadRateDefault(u8 br)
{
	/* config UART_RXD for output mode */
	/* For configure BC7701 buad rate*/
	AFIO_GPxConfig(UART_TXD_AFIO_PORT, UART_TXD, AFIO_FUN_GPIO);
	AFIO_GPxConfig(UART_RXD_AFIO_PORT, UART_RXD, AFIO_FUN_GPIO);
   /* configure TXD & RXD pull-up function enabled  */
	GPIO_PullResistorConfig(UART_TXD_PORT,UART_TXD,GPIO_PR_UP);
	GPIO_PullResistorConfig(UART_RXD_PORT,UART_RXD,GPIO_PR_UP);

	GPIO_DirectionConfig(UART_TXD_PORT,UART_TXD,GPIO_DIR_IN);
	GPIO_DirectionConfig(UART_RXD_PORT,UART_RXD,GPIO_DIR_OUT);
	/* For configure BC7701 buad rate*/
	/* BC7701 TXD=1 & RXD=1, buad rate = 115200 */
	/* BC7701 TXD=1 & RXD=0, buad rate = 9600 */
	if(br == BAUD_RATE_9600)
		UART_RXD_PORT->RR = UART_RXD;		//MCU RXD->BC7701 TXD=0
	else
		UART_RXD_PORT->SRR = UART_RXD;   //MCU RXD->BC7701 TXD=1
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_HardwareBuadRateRelease
* Description   : TXD & RXD release to UART
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_HardwareBuadRateRelease(void)
{
	/* Config AFIO mode as UART_RXD and UART_TXD function. */
	AFIO_GPxConfig(UART_TXD_AFIO_PORT, UART_TXD, UART_TXD_AFIO_FUN);
	AFIO_GPxConfig(UART_RXD_AFIO_PORT, UART_RXD, UART_RXD_AFIO_FUN);
   /* configure TXD & RXD pull-up function disable  */
	GPIO_PullResistorConfig(UART_TXD_PORT,UART_TXD,GPIO_PR_DISABLE);
	GPIO_PullResistorConfig(UART_RXD_PORT,UART_RXD,GPIO_PR_DISABLE);

}
/*------------------------------------------------------------------------------
* Function Name : BC7701_HardwareReset
* Description   : hardware reset BC7701
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_HardwareReset(void)
{
	RESET_PIN_PORT->RR = RESET_PIN;
	delay_ms(4);
	RESET_PIN_PORT->SRR = RESET_PIN;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SoftwareReset
* Description   : Reset command to BC7701
* Input         : None
* Output        : None
* Return        : BCI packet index
------------------------------------------------------------------------------*/
void *BC7701_SoftwareReset(void)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(u32));
	if(p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(u32);			//flag+opcode+parameter=7
		p->flag = 0x00;
		p->opcode = BCI_SOFT_RESET;
		p->param[0]=0;
		p->param[1]=0;
		p->param[2]=0;
		p->param[3]=0;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SoftwareResetKeep
* Description   : Reset command to BC7701 & keep keep pervious parameter
* Input         : None
* Output        : None
* Return        : BCI packet index
------------------------------------------------------------------------------*/
void *BC7701_SoftwareResetKeep(void)
{
	tBCI_PACKAGE *p;
	p = malloc(BCI_HEAD_LENG+2*sizeof(u32));
	if(p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+2*sizeof(u32);			//flag+opcode+parameter=11
		p->flag = 0x00;
		p->opcode = BCI_SOFT_RESET;
		p->param[0]=0x32;
		p->param[1]=0x30;
		p->param[2]=0x32;
		p->param[3]=0x36;
		p->param[4]=0x00;
		p->param[5]=0x03;
		p->param[6]=0x00;
		p->param[7]=0x00;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SendHCIPackage
* Description   : send HCI command package to BC7701
* Input         :
*				- opcode : HCI command
*				- len : parameter data length
*				- pbuf : HCI parameter index
* Output        : None
* Return        : HCI package index
------------------------------------------------------------------------------*/
void *BC7701_SendHCIPackage(u16 opcode,u8 len,u8 *pbuf)
{
	tHCI_CMD_PKG *p;

	p = malloc(len+4);
	if(p != NULL)
	{
		p->type = HCI_CMD_PKG;
		p->opcode = opcode;
		p->length = len;
		if((len != 0) && (pbuf != NULL))	memcpy(p->param,pbuf,len);
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SendBCIPackage
* Description   : send BCI command package to BC7701
* Input         :
*				- opcode : BCI command
*				- flag : BCI package flag
*				- len : parameter data length
*				- pbuf : BCI parameter index
* Output        : None
* Return        : BCI package index
------------------------------------------------------------------------------*/
void *BC7701_SendBCIPackage(u16 opcode,u8 flag,u8 len,u8 *pbuf)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+len);
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+len;		//flag+opcode=3byte
		p->flag = flag;
		p->opcode = opcode;
		if((len != 0) && (pbuf != NULL))	memcpy(p->param,pbuf,len);
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetDeviceName
* Description   : set BLE device name
* Input         :
*				- leng : device name length(max 31byte)
*				- name : device name string
* Output        : None
* Return        : BCI package index
------------------------------------------------------------------------------*/
void *BC7701_SetDeviceName(u8 leng,u8 *name)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+leng);
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+leng;	// flag+opcode+parameter=leng+3
		p->flag = 0x00;
		p->opcode = BCI_DEV_NAME;
		if((leng != 0) && (name != NULL)) memcpy(p->param,name,leng);
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetDeviceAddress
* Description   : set BLE device address(MAC address)
* Input         :
*				- adr : device address(6 byte)
*				- type : device address type
*					- 0x00 : static address type
*					- 0x01 : random address type
* Output        : None
* Return        : BCI package index
------------------------------------------------------------------------------*/
void *BC7701_SetDeviceAddress(u8 *adr,u8 type)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(tDEV_ADDR));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(tDEV_ADDR);		// flag+opcode+parameter=10
		p->flag = 0x00;
		p->opcode = BCI_DEV_ADDRESS;
		if(adr != NULL) memcpy(((tDEV_ADDR *)(p->param))->adr,adr,6);
		((tDEV_ADDR *)(p->param))->type = type;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_ConnectIntervalModify
* Description   : modify connect interval
* Input         :
*				- opcode : BCI_CONN_INTV or BCI_CONN_INTV1
*				- min : Minimum time connections interval(unit 1.25ms)
*				- max : Maximum time connection interval(BCI_CONN_INTV1 only)(unit 1.25ms)
* Output        : None
* Return        : BCI package index
------------------------------------------------------------------------------*/
void *BC7701_ConnectIntervalModify(u16 opcode,u16 min,u16 max)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(tCNNT_INTV));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->flag = 0x00;
		if(opcode == BCI_CONN_INTV)
		{
			p->opcode = BCI_CONN_INTV;
			p->length = BCI_COMD_LENG+sizeof(u16);			// flag+opcode+parameter=5
			((tCNNT_INTV *)(p->param))->min = min;
		}
		else
		{
			p->opcode = BCI_CONN_INTV1;
			p->length = BCI_COMD_LENG+sizeof(tCNNT_INTV); // flag+opcode+parameter=11
			((tCNNT_INTV *)(p->param))->min = min;
			((tCNNT_INTV *)(p->param))->max = max;
			((tCNNT_INTV *)(p->param))->latency = 0;
			((tCNNT_INTV *)(p->param))->timeout = 300;
		}
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetAdvertisingData
* Description   : set Advertising data
* Input         :
*				- mode : How to set up Advertising data
*					- ADV_JOIN_NAME 	: Automatically join device name(DEV_Name command)
*					- ADV_UNJOIN_NAME : unjoin device name
*				- leng : Advertising data length(max 31 bytes)
*				- advdata : advertising data
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetAdvertisingData(u8 mode,u8 leng,u8 *advdata)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+leng);
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+leng;	// flag+opcode+parameter=leng+3
		p->flag = mode;
		p->opcode = BCI_ADV_DATA;
		if((leng != 0) && (advdata != NULL)) memcpy(p->param,advdata,leng);
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetScanResponseData
* Description   : Set Scan Response Data
* Input         :
*				- leng : Scan Response Data length(max 31 bytes)
*				- sdata : Scan Response Data
*				- Note : If leng=0 or sdata=NULL,
*							Scan Response Data will be set to none
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetScanResponseData(u8 leng,u8 *sdata)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+leng);
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+leng;	// flag+opcode+parameter=leng+3
		p->flag = 0x00;
		p->opcode = BCI_SCAN_DATA;
		if((leng != 0) && (sdata != NULL))
		{
			memcpy(p->param,sdata,leng);
		}
		else
		{
			p->length = BCI_COMD_LENG+1;	// flag+opcode+parameter=4
			p->param[0] = 0x00;
		}
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_AdvertisingInterval
* Description   : set Advertising interval to BC7701
* Input         :
*				- min : Minimum time advertising interval(unit 0.625ms)
*				- max : Maximum time advertising interval(unit 0.625ms)
*				- chmap : advertising channel
*					- b0 : channel 37(2402MHz)
*					- b1 : channel 38(2426MHz)
*					- b2 : channel 39(2480Mhz)
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_AdvertisingInterval(u16 min,u16 max,u8 chmap)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(tADV_INTV));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(tADV_INTV);	// flag+opcode+parameter=8
		p->flag = 0x00;
		p->opcode = BCI_ADV_INTV;
		((tADV_INTV *)(p->param))->min = min;
		((tADV_INTV *)(p->param))->max = max;
		((tADV_INTV *)(p->param))->map = chmap & 0x07;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_AdvertisingControl
* Description   : control BC7701 enable or disable Advertising function
* Input         :
*				- ctrl : ENABLE or DISABLE Advertising
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_AdvertisingControl(ControlStatus ctrl)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(u8));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(u8);	// flag+opcode+parameter=4
		p->flag = 0x00;
		p->opcode = BCI_ADV_CTRL;
		p->param[0] = (ctrl == ENABLE) ? 1 : 0;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetTxPower
* Description   : set BC7701 Tx Power
* Input         :
*				- pwr : Tx Power level 0x00~0x0F
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetTxPower(u8 pwr)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(u8));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(u8);	// flag+opcode+parameter=4
		p->flag = 0x00;
		p->opcode = BCI_TX_POWER;
		p->param[0] = pwr & 0x0F;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetCrystalCload
* Description   : set BC7701 crystal C-Load level
* Input         :
*				- cc : crystal C-Load level 0x00~0x0F
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetCrystalCload(u8 cc)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(u8));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(u8);	// flag+opcode+parameter=4
		p->flag = 0x00;
		p->opcode = BCI_CLOAD_SET;
		p->param[0] = cc & 0x0F;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetupFeatureFlag
* Description   : setup feature flag
* Input         :
*				- md : update feature flag mode
*					-FEATURE_DIR : Direct update feature flag
*					-FEATURE_SET : set feature flag
*					-FEATURE_CLR : clear feature flag
*					-Others : no action
*				- sff : update feature flag data
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetupFeatureFlag(u8 md,FeatureFlag sff)
{
	tBCI_PACKAGE *p;

	switch(md)
	{
		case FEATURE_DIR : md = 0x00;	break;
		case FEATURE_SET : md = 0x10;	break;
		case FEATURE_CLR : md = 0x20; sff = ~sff; break;
		default :	//others mode no action
			return(NULL);
	}

	p = malloc(BCI_HEAD_LENG+sizeof(tFEATURE));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(tFEATURE);	// flag+opcode+parameter=7
		p->flag = md;
		p->opcode = BCI_FEATURE_CTRL;
//		((tFEATURE *)(p->param))->u = sff;
		memcpy(p->param,&sff,sizeof(FeatureFlag));
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetOperateMode
* Description   : set BC7701 operate mode
* Input         :
*				- omd : operate mode
*					-OP_NORMAL : normal mode
*					-OP_DEEPSLEEP : deep sleep mode
*					-OP_POWERDOWN : power down mode(Only hardware reset)
*					-Others : No action
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetOperateMode(u8 omd,ControlStatus ctrl,u8 wuw,u8 wut)
{
	tBCI_PACKAGE *p;

	switch(omd)
	{
		case OP_NORMAL :		omd = 0x00;	break;
		case OP_DEEPSLEEP :	omd = 0x01;	break;
		case OP_POWERDOWN :  omd = 0x15;	break;
		default :	//others mode no action
			return(NULL);
	}
	if(ctrl == ENABLE)	p = malloc(BCI_HEAD_LENG+sizeof(u8)*8);
	else                 p = malloc(BCI_HEAD_LENG+sizeof(u8));
	if(p != NULL)
	{
		p->type = BCI_CMD_PKG;
		if(ctrl == ENABLE)	p->length = BCI_COMD_LENG+sizeof(u8)*8;	// flag+opcode+parameter=4
		else                 p->length = BCI_COMD_LENG+sizeof(u8);		// flag+opcode+parameter=4
		p->flag = 0x00;
		p->opcode = BCI_OPERATE_MODE;
		p->param[0] = omd;
		if(ctrl == ENABLE)
		{
			p->param[1] = 0;
			p->param[2] = 0;
			p->param[3] = 0;					//int32k

			p->param[4] = 1;              //master wake up enable
			p->param[5] = wut;				//master wake up delay time
			p->param[6] = wuw;            //master wake up width
			p->param[7] = 0;
		}
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetWhiteList
* Description   : set BC7701 white list for MAC address & erase before white list
* Input         :
*				- erase : Enable or Disable erase before white list
*					-ENABLE : erase before white list
*					-DISABLE: no erase
*				- adr : Add whitelisted address
*				- mask : Add the address mask of the whitelist
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetWhiteList(ControlStatus erase,u8 *adr,u8 *mask)
{
	tBCI_PACKAGE *p;

	p = malloc(BCI_HEAD_LENG+sizeof(tWHITE_LIST));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(tWHITE_LIST);	// flag+opcode+parameter=15
		if(erase == ENABLE)
			p->flag = 0x10;	//erase before white list
		else
			p->flag = 0x00;
		p->opcode = BCI_WHITE_LIST;
		memcpy(((tWHITE_LIST *)(p->param))->adr,adr,6);
		memcpy(((tWHITE_LIST *)(p->param))->mask,mask,6);
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetBaudRate
* Description   : set BC7701 UART interface baud rate
* Input         :
*				- br : baud rate type
*					- BAUD_RATE_2400 : baud rate 2400bps
*					- BAUD_RATE_4800 : baud rate 4800bps
*					- BAUD_RATE_9600 : baud rate 9600bps
*					- BAUD_RATE_14400 : baud rate 14400bps
*					- BAUD_RATE_19200 : baud rate 19200bps
*					- BAUD_RATE_38400 : baud rate 38400bps
*					- BAUD_RATE_57600 : baud rate 57600bps
*					- BAUD_RATE_115200 : baud rate 115200bps
*					- BAUD_RATE_125K : baud rate 125000bps
* Output        : None
* Return        : BCI package index,NULL for fail
------------------------------------------------------------------------------*/
void *BC7701_SetBaudRate(u8 br)
{
	tBCI_PACKAGE *p;

	if(br > BAUD_RATE_125K) return(NULL);
	p = malloc(BCI_HEAD_LENG+sizeof(u32));
	if( p != NULL)
	{
		p->type = BCI_CMD_PKG;
		p->length = BCI_COMD_LENG+sizeof(u32);	// flag+opcode+parameter=leng+3
		p->flag = 0x00;
		p->opcode = BCI_BAUDRATE;
//		*((u32 *)p->param) = BaudRateTable[br];
		memcpy(p->param,&BaudRateTable[br],sizeof(u32));
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_DummyWakeup
* Description   : send 0x00 data to wake up BC7701
* Input         :
*				- len : send 0x00 data length
* Output        : None
* Return        : 
------------------------------------------------------------------------------*/
bool BC7701_DummyWakeup(void)
{
	void *p;
	p = malloc(sizeof(DummyWakeupPackage));
	if(p != NULL)
	{
		memcpy(p,DummyWakeupPackage,sizeof(DummyWakeupPackage));
		((u8 *)p)[1] = DummyWakeUpWidth[UART_BaudRate];
		if(BC7701_TransmitPackage(p)) return(TRUE);
		else free(p);
	}
	return(FALSE);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_TransmitPackage
* Description   : send package data to BC7701
* Input         : package index
* Output        : None
* Return        :
*				- bool flag
*					- TRUE : package index to Queue
*					- FALSE : Queue buffer full
------------------------------------------------------------------------------*/
bool BC7701_TransmitPackage(void *pbuf)
{
	if(IS_TXD_QUE_FULL(1)) return(FALSE);
	TxSerialQueue[uTxQWriteIdx] = pbuf;
	if(++uTxQWriteIdx == TXSERIAL_QUEUE_MAX) uTxQWriteIdx = 0;
	UART_PORT->IER |= USART_INT_TXDE;
	return(TRUE);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_ReadTransmitEmpty
* Description   : Get Transmit Queue status
* Input         : None
* Output        : None
* Return        :
*				- bool flag
*					- TURE : Transmit Queue is empty
*					- FALSE : Transmit Queue not empty
------------------------------------------------------------------------------*/
bool BC7701_ReadTransmitEmpty(void)
{
	if(IS_TXD_QUE_EMPTY() && (uTxPackageIdx == NULL)) return(TRUE);
	return(FALSE);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_ReadReceiveEmpty
* Description   : Get receive status
* Input         : None
* Output        : None
* Return        :
*				- bool flag
*					-TRUE : no receive data parser
*					-FALSE : parser receive
------------------------------------------------------------------------------*/
bool BC7701_ReadReceiveEmpty(void)
{
	if(IS_RXD_BUF_EMPTY() && (uRxPkgStep == PARS_PKG_TYPE)) return(TRUE);
	return(FALSE);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_ReadReceivePackage
* Description   : Read receive package
* Input         : None
* Output        : None
* Return        :
* 				- package index
------------------------------------------------------------------------------*/
void *BC7701_ReadReceivePackage(void)
{
	void *p = NULL;

	if(!IS_RXD_QUE_EMPTY())
	{
		p = RxSerialQueue[uRxQReadIdx];
		if(++uRxQReadIdx == RXSERIAL_QUEUE_MAX) uRxQReadIdx = 0;
	}
	return(p);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_WriteReceivePackage
* Description   : write receive package index to queue
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_WriteReceivePackage(void)
{
	if(IS_RXD_QUE_FULL(1))
	{
		free(uRxPkgIndex);
	}
	else
	{
		RxSerialQueue[uRxQWriteIdx] = uRxPkgIndex;
		if(++uRxQWriteIdx == RXSERIAL_QUEUE_MAX)	uRxQWriteIdx = 0;
	}
	uRxPkgStep = PARS_PKG_TYPE;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_WriteReceivePackage
* Description   : write receive package index to queue
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_ReceiveParserPackage(void)
{
	u8	uData;

   while(!IS_RXD_BUF_EMPTY())
   {	
		uData = uRxSerialBuffer[uRxSBReadIdx];
		if(++uRxSBReadIdx >= UART_BUFFER_SIZE) uRxSBReadIdx = 0;		
		switch(uRxPkgStep)
		{
			case PARS_EVENT_PARAM :
				uRxPkgIndex[uRxPkgPosint++] = uData;
				if(--uRxPkgLength == 0)	BC7701_WriteReceivePackage();
				break;
			case PARS_HCI_EVENT :
				uRxPkgEvent = uData;
				uRxPkgStep = (uRxPkgEvent == EVENT_CMD_COMPLETE) ? PARS_EVENT_LENG : PARS_PKG_TYPE;
				break;
			case PARS_EVENT_LENG :
				uRxPkgLength = uData;
				//check package size if=0 or >max size to error
				switch(uRxPkgType)
				{
					case HCI_EVENT_PKG:
						if((uRxPkgLength < 4) || (uRxPkgLength > 128))
						{
							uRxPkgStep = PARS_PKG_TYPE;
							break;
						}
						uRxPkgIndex = malloc(uRxPkgLength+3);
						if(uRxPkgIndex != NULL)
						{
							uRxPkgIndex[0] = uRxPkgType;
							uRxPkgIndex[1] = uRxPkgEvent;
							uRxPkgIndex[2] = uRxPkgLength;
							uRxPkgPosint	= 3;
							uRxPkgStep = PARS_EVENT_PARAM;
						}
						else
						{							
							uRxPkgStep = PARS_PKG_TYPE;
						}
						break;
					case BCI_EVENT_PKG:
						if((uRxPkgLength < BCI_COMD_LENG) || (uRxPkgLength > (128+BCI_COMD_LENG)))
						{
							uRxPkgStep = PARS_PKG_TYPE;
							break;
						}
						uRxPkgIndex = malloc(uRxPkgLength+BCI_TYPE_LENG);
						if(uRxPkgIndex != NULL)
						{
							uRxPkgIndex[0] = uRxPkgType;
							uRxPkgIndex[1] = uRxPkgLength;
							uRxPkgPosint	= BCI_TYPE_LENG;
							uRxPkgStep = PARS_EVENT_OPCODE;
						}
						else
						{
							uRxPkgStep = PARS_PKG_TYPE;
						}
						break;
				}
				break;
			case PARS_EVENT_OPCODE :
				uRxPkgIndex[uRxPkgPosint++] = uData;
				uRxPkgLength--;
				if(uRxPkgPosint == 5)
				{
					//check opcode 0x00XX,0xFFXX,0x18XX,0x2AXX
					//0x00XX --> BCI Command
					//0xFFXX --> Unknow Service
					//0x18XX --> profile Service UUID
					//0x2AXX --> Characteristic UUID
					if((uRxPkgIndex[4] == 0x00) || (uRxPkgIndex[4] == 0xFF) ||
						(uRxPkgIndex[4] == 0x18) || (uRxPkgIndex[4] == 0x2A) 	)
					{
						if(uRxPkgLength > 0) uRxPkgStep = PARS_EVENT_PARAM;
						else	BC7701_WriteReceivePackage();
					}
					else
					{
						free(uRxPkgIndex);
						uRxPkgStep = PARS_PKG_TYPE;
					}
				}
				break;
			case PARS_PKG_TYPE :
				switch(uData)
				{
					case HCI_EVENT_PKG :             	/* 0x04 */
						uRxPkgType = HCI_EVENT_PKG;
						uRxPkgLength = 2;          		/* event code=1byte + data length=1byte */
						uRxPkgStep = PARS_HCI_EVENT; 	/* to receive event code data step */
						break;
					case BCI_EVENT_PKG :						/* 0x78 */
						uRxPkgType = BCI_EVENT_PKG;
						uRxPkgLength = 1;          		/* data length=1byte */
						uRxPkgStep = PARS_EVENT_LENG; 	/* to receive length data step */
						break;
				}
				break;
		}
	}
}
/*------------------------------------------------------------------------------
* Function Name : UART_IRQHandler
* Description   : UART interruption processed
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void UART_IRQHandler(void)
{
	if(UART_PORT->SR & USART_FLAG_RXDR)	
	{
		uRxSerialBuffer[uRxSBWriteIdx++] = UART_PORT->DR; 
		if(uRxSBWriteIdx == UART_BUFFER_SIZE) uRxSBWriteIdx = 0;
	}

	if(((UART_PORT->SR) & USART_FLAG_TXDE))
	{
		if(uTxPackageIdx == NULL)
		{
			if(IS_TXD_QUE_EMPTY())
			{
				UART_PORT->IER &= ~USART_INT_TXDE;
			}
			else
			{
				uTxPackageIdx = TxSerialQueue[uTxQReadIdx];
				uTxPackagePos = 0;
				switch(((u8 *)uTxPackageIdx)[0])
				{
					case HCI_CMD_PKG :
						uTxPackageLength = ((tHCI_CMD_PKG *)uTxPackageIdx)->length;
						/* type+opcode+length = 4byte */
						uTxPackageLength += 4;
						break;
					case BCI_CMD_PKG :
						uTxPackageLength = ((tBCI_PACKAGE *)uTxPackageIdx)->length;
						/* type+length = 2byte */
						uTxPackageLength += 2;
						break;
					case 0xFF :	// Dummy Wakeup Package
						uTxPackageLength = ((u8 *)uTxPackageIdx)[1];
						uTxPackagePos = 2;
						break;
				}
				if(++uTxQReadIdx == TXSERIAL_QUEUE_MAX) uTxQReadIdx = 0;
			}
		}
		if(uTxPackageIdx != NULL)
		{
			UART_PORT->DR = ((u8 *)uTxPackageIdx)[uTxPackagePos++];
			if(--uTxPackageLength == 0)
			{
				free(uTxPackageIdx);
				uTxPackageIdx = NULL;
			}
		}
	}

	/*--------------------------------------------------------------------------------------------------------*/
	/* DSB instruction is added in this function to ensure the write operation which is for clearing interrupt*/
	/* flag is actually completed before exiting ISR. It prevents the NVIC from detecting the interrupt again */
	/* since the write register operation may be pended in the internal write buffer of Cortex-Mx when program*/
	/* has exited interrupt routine. This DSB instruction may be masked if this function is called in the     */
	/* beginning of ISR and there are still some instructions before exiting ISR.                             */
	/*--------------------------------------------------------------------------------------------------------*/
	__DSB();
}
